/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	pgpMiscNT.c	- miscellaneous NT-specific functions for PGP Utility Driver
	

	$Id: pgpMiscNT.c,v 1.4 2001/03/12 21:24:40 pbj Exp $
____________________________________________________________________________*/

#include "ntddk.h"
#include "stdarg.h"

#ifndef PGP_EROS
#define PGP_EROS	0
#endif // PGP_EROS

#include "pgpMisc.h"

// typedefs
typedef struct {
	KSPIN_LOCK		spinlock;
	KIRQL			irql;
	FAST_MUTEX		mutex;
} pgpDriverCriticalSection;


//	______________________________________________________
//
//  memory allocation

PVOID
pgpDriverSecureAlloc (
		ULONG	ulNumBytes)
{
	return (ExAllocatePool (NonPagedPool, ulNumBytes));
}

VOID
pgpDriverSecureFree (
		PVOID p)
{
	ExFreePool (p);
}


//	______________________________________________________
//
//  critical section handlers

VOID
pgpDriverEnterCriticalSection (
		pgpDriverCriticalSection*	pSection,
		ULONG						ulType)
{
	if (ulType == SPINLOCK)
		KeAcquireSpinLock (&pSection->spinlock, &pSection->irql);
	else
		ExAcquireFastMutex (&pSection->mutex);

}

ULONG
pgpDriverTryToEnterCriticalSection (
		pgpDriverCriticalSection*	pSection,
		ULONG						ulType)
{
	if (ulType == SPINLOCK)
	{
		KeAcquireSpinLock (&pSection->spinlock, &pSection->irql);
		return TRUE;
	}
	else
	{
		return ExTryToAcquireFastMutex (&pSection->mutex);
	}
}

VOID
pgpDriverLeaveCriticalSection (
		pgpDriverCriticalSection*	pSection,
		ULONG						ulType)
{
	if (ulType == SPINLOCK)
		KeReleaseSpinLock (&pSection->spinlock, pSection->irql);
	else
		ExReleaseFastMutex (&pSection->mutex);

}


VOID
pgpDriverInitCriticalSection (
		pgpDriverCriticalSection*	pSection)
{
	KeInitializeSpinLock (&pSection->spinlock);
	ExInitializeFastMutex (&pSection->mutex);
}

VOID
pgpDriverCleanupCriticalSection (
		pgpDriverCriticalSection*	pSection)
{

}

void
pgpCopyMemory(void *src, void *dst, int size)
{
	memcpy(dst, src, size);
}


#if !PGP_EROS
//	______________________________________________________
//
//  high resolution timing

// RDTSC instruction is not supported by Visual C++ inline assembler
// so we use this #define to fake it
#define rdtsc __asm __emit 0fh __asm __emit 031h

// Tests for presence of Pentium Time Stamp Counter high resolution
// timer by executing the RDTSC instruction.  If the processor does
// not support this instruction, an exception is generated and this
// function returns FALSE.

static ULONG
sTestForHiResTimer (VOID)
{
	ULONG	present = TRUE;

	__try
	{
		__asm 
		{
			rdtsc
		}
	}

	__except (EXCEPTION_EXECUTE_HANDLER) 
	{
		present = FALSE;
	}

	return present;
}

/*
 * Add as much timing-dependent random noise as possible
 * to the randPool.  Typically, this involves reading the most
 * accurate system clocks available.
 *
 * Returns the number of ticks that have passed since the last call,
 * for entropy estimation purposes.
 */

VOID
pgpDriverGetHiResTime (
		PULONG	pulLowPart)
{
	static ULONG tsctested = FALSE;
	static ULONG tscpresent = FALSE;

	if (!tsctested) {
		tscpresent = sTestForHiResTimer ();
		tsctested = TRUE;
	}

	if (tscpresent) 
	{
		ULONG tsclow;
		__asm 
		{
			rdtsc
			mov		tsclow, eax
		}
		*pulLowPart = tsclow;
	}
	else 
	{
		LARGE_INTEGER	time;

		KeQuerySystemTime (&time);

		*pulLowPart = time.LowPart;
	}
}
#endif // !PGP_EROS
